#ifndef _lf_xbar32_h_
#define _lf_xbar32_h_

#include "lf_fabric.h"
#include "libmyri.h"

/*
 * Hardware-specific struct for a tagged xbar
 */

/* magic route byte for querying a tagged xbar */
#define LF_TAGGED_XBAR_QUERY_ROUTE 0xA0

/*
 * This struct must start 4 bytes into a message and be clear to zero
 * for CRC to be computed properly
 */
struct lf_tagged_xbar_insert {
  uint32_t id;
  uint8_t absolute_port;
  uint8_t quadrant_disabled;
  uint8_t xbar_size;
  uint8_t pad0;
  uint32_t status_mask;
  uint32_t pad1;
  uint32_t crc32;
};

static inline int
lf_xbar_qd_violation(
  struct lf_xbar *xp,
  int in_port,
  int out_port)
{
  int half;

  /* does not apply to non-tagged xbars */
  if (xp->xbar_id == 0) {
    return FALSE;
  }

  half = xp->num_ports / 2;

  if ((xp->quadrant_disable & 0x01) && in_port < half && out_port < half) {
    return TRUE;
  }
  if ((xp->quadrant_disable & 0x02) && in_port >= half && out_port >= half) {
    return TRUE;
  }
  return FALSE;
}

static inline int
lf_xbar_num_ports(
  enum myri_line_speed speed,
  int xbar_size)
{
  if (speed == MYRI_SPEED_2G) {
    return 32;
  } else if (xbar_size == 0) {
    return 16;
  } else {
    return 32;
  }
}

/*
A packet requesting an ID from a xbar will have the following format:

[0xA0] - a head byte for "-32"a
[route] - any number of bytes for the return route
[type] - 4 bytes for the packet type.
[reserved] - 20 zero bytes
[info] - any number of bytes for the mapper packet info
[CRC32] - 4 bytes for the CRC32
[CRC8] - 1 byte for the CRC 8

The xbar will fill in the 20 zero bytes as follows:

4 bytes for the ID = (serial number)*16 + (xbar number)
1 byte for the absolute port number
1 byte for quadrant disable (see below)
1 byte for xbar size
1 byte reserved
4 bytes port status (see below)
4 bytes reserved (for xbar64 port status)
4 bytes for CRC32 correction.

Quadrant disable:
0x01 indicates if the lower half of the xbar is enabled/disabled
0x02 indicates if the upper half of the xbar is enabled/disabled

Lower half = 0-7 of a xbar16, 8-15 of a xbar32.
Upper half = 8-15 of a xbar16, 16-31 of a xbar32.

port status:
one bit per port. 1 means port is up, 0 means port is not up.
LSB of first byte is for port 0.

****************
for 10G Xbar32, the xbar size byte will be used to
indicate how big the Xbar is.
0x00 = xbar16
0x01 = xbar32
*/

#endif /* _lf_xbar32_h_ */
